﻿using DataAccess;
using Newtonsoft;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Xml;
using System.Xml.Linq;


/// <summary>
/// Summary description for CInstrumentResponse
/// </summary>
public class CInstrumentResponse
{
    public BaseMaster m_BaseMstr { set; get; }

    public CInstrumentResponse(BaseMaster BaseMstr)
	{
        m_BaseMstr = BaseMstr;
	}

    /// <summary>
    /// US:5768 US:6186
    /// Write instrument responses in the TBICDS central database
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lMID"></param>
    /// <param name="lTID"></param>
    /// <param name="lQID"></param>
    /// <param name="lRID"></param>
    /// <param name="lScoreValue"></param>
    /// <param name="strResponseValue"></param>
    /// <returns></returns>
    public bool InsertInstrumentResponse(string strPatientID,
                                        long lIntakeID,
                                        long lMID,
                                        long lTID,
                                        long lQID,
                                        long lRID,
                                        long lScoreValue,
                                        string strResponseValue) 
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nTID", lTID);
        plist.AddInputParameter("pi_nQID", lQID);
        plist.AddInputParameter("pi_nRID", lRID);
        plist.AddInputParameter("pi_nScoreValue", lScoreValue);
        plist.AddInputParameter("pi_vResponseValue", strResponseValue);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.InsertInstrumentResponse",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// US:6186 US:6247
    /// Get the instrument's responses dataset
    /// </summary>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public DataSet GetInstrumentResponsesDS(long lMID, long lIntakeID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);
        /*
            pi_vPatientID      in varchar2,
            pi_nSiteID         in number,
            pi_nClinicalProgID in number,
            pi_nPathwayID      in number,
            pi_nMID            in number,
            pi_nIntakeID       in number,
         */

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", m_BaseMstr.SelectedPatientID);

        //plist.AddInputParameter("pi_nSiteID", 1);
        //plist.AddInputParameter("pi_nClinicalProgID", 1);
        //plist.AddInputParameter("pi_nPathwayID", 1);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);

        //
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetInstrumentResponsesRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;
        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6186 US:6247
    /// Get a JSON string of the instrument responses
    /// </summary>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public string GetInstrumentResponsesJson(long lMID, long lIntakeID) 
    {
        string strJSON = "null";

        DataSet ds = this.GetInstrumentResponsesDS(lMID, lIntakeID);
        if(ds != null){
            strJSON = JsonConvert.SerializeObject(ds.Tables[0], Newtonsoft.Json.Formatting.Indented);
        }

        return strJSON;
    }

    /// <summary>
    /// Get a dataset of resolved logigal variables to be used in the scoring
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public DataSet GetIntakeLogicVarDS(string strPatientID, long lMID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_vKey", m_BaseMstr.Key);
        plist.AddInputParameter("pi_nMID", lMID);

        //
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetIntakeLogicVarRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;
        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// Get the score logic to be used to calculate the instrument score and interpretation
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <returns></returns>
    protected DataSet GetScoreLogicDS(long lMID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_nMID", lMID);

        //
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetScoreLogicRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;
        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// Return a string with the score equation for the selected instrument
    /// </summary>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public string GetScoreLogic(long lMID) 
    {
        string strLogic = String.Empty;
        DataSet ds = this.GetScoreLogicDS(lMID);
        if (ds != null) {
            foreach (DataTable dt in ds.Tables) {
                foreach (DataRow dr in dt.Rows) {
                    if (!dr.IsNull("SCORE_LOGIC")) {
                        strLogic = dr["SCORE_LOGIC"].ToString();
                    }
                }
            }
        }
        return strLogic;
    }

    /// <summary>
    /// Create a responses dataset to pass to the COM componet for scoring
    /// </summary>
    /// <param name="lsResponses"></param>
    /// <returns></returns>
    public DataSet ConvertResponsesToDataSet(List<IntakeResponse> lsResponses) 
    {
        
        //create the new dataset for the responses
        DataSet ds = new DataSet("IntakeResponses");
        ds.Tables.Add(new DataTable
        {
            TableName = "Responses",
            Columns = { 
                new DataColumn("MID", typeof(long)),
                new DataColumn("TID", typeof(long)),
                new DataColumn("QID", typeof(long)),
                new DataColumn("RID", typeof(long)),
                new DataColumn("IDENTIFIER", typeof(string)),
                new DataColumn("RESPONSE_VALUE", typeof(string)),
                new DataColumn("SCORE_VALUE", typeof(long)),
                new DataColumn("RESPONSE_TYPE", typeof(long))
            }
        });

        //iterate the list of response objects
        foreach (IntakeResponse r in lsResponses) {
            DataRow dr = ds.Tables[0].NewRow();
            dr["MID"] = r.MID;
            dr["TID"] = r.TID;
            dr["QID"] = r.QID;
            dr["RID"] = r.RID;
            dr["IDENTIFIER"] = r.Identifier;
            dr["RESPONSE_VALUE"] = r.ResponseValue;
            dr["SCORE_VALUE"] = r.ScoreValue;
            dr["RESPONSE_TYPE"] = r.ResponseType;

            ds.Tables[0].Rows.Add(dr);
        }

        ds.AcceptChanges();

        return ds;
    }

    // ----------------------------

    /// <summary>
    /// US:6247 US:6186
    /// Gets and returns a dataset of responses for the completed instrument
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public DataSet GetInstrumentResponsesDS(string strPatientID, long lMID, long lIntakeID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        /*
            pi_vPatientID      in varchar2,
            pi_nSiteID         in number,
            pi_nClinicalProgID in number,
            pi_nPathwayID      in number,
            pi_nMID            in number,
            pi_nIntakeID       in number,
         */

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        //plist.AddInputParameter("pi_nSiteID", 0);
        //plist.AddInputParameter("pi_nClinicalProgID", 0);
        //plist.AddInputParameter("pi_nPathwayID", 0);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);

        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetInstrumentResponsesRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }

    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public DataSet GetMergedInstrumentResponsesDS(string strPatientID, long lMID, long lIntakeID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);

        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetMergedInstrumentResponsesRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }

    }


    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <returns></returns>
    public DataSet GetPatientInstrumentsDS(string strPatientID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);

        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetPatientInstrumentsRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }

    }

    /// <summary>
    /// US:6247 US:6186
    /// Returns a JSON object with the responses of the completed instrument
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public object GetInstrumentResponsesJSON(string strPatientID, long lMID, long lIntakeID)
    {
        var obj = new object { };

        //get the responses dataset
        DataSet ds = this.GetInstrumentResponsesDS(strPatientID, lMID, lIntakeID);
        if (ds != null)
        {
            if (ds.Tables[0].Rows.Count > 0)
            {

                long m_lMID = 0;
                long m_lTID = 0;
                long m_lQID = 0;

                //instantiate the module object
                var objModule = new ModelIntake.Module
                {
                    MID = m_lMID,
                    ModuleTitle = String.Empty,
                    Topics = new List<ModelIntake.Topic>()
                };

                //loop the dataset
                foreach (DataTable dt in ds.Tables)
                {
                    foreach (DataRow dr in dt.Rows)
                    {

                        //update the module object
                        if (Convert.ToInt32(dr["mid"]) != m_lMID)
                        {
                            m_lMID = Convert.ToInt32(dr["mid"]);

                            objModule.MID = m_lMID;
                            objModule.ModuleTitle = dr["module"].ToString();
                        }

                        //create the topics node
                        if (Convert.ToInt32(dr["tid"]) != m_lTID)
                        {
                            m_lTID = Convert.ToInt32(dr["tid"]);

                            objModule.Topics.Add(new ModelIntake.Topic
                            {
                                TID = m_lTID,
                                TopicTitle = dr["topic"].ToString(),
                                Questions = new List<ModelIntake.Question>()
                            });
                        }

                    }
                }

                // add questions to the topics
                foreach (ModelIntake.Topic t in objModule.Topics)
                {

                    //loop the dataset
                    foreach (DataTable dt in ds.Tables)
                    {
                        foreach (DataRow dr in dt.Rows)
                        {
                            if (Convert.ToInt32(dr["tid"]) == t.TID)
                            {
                                if (Convert.ToInt32(dr["qid"]) != m_lQID)
                                {
                                    m_lQID = Convert.ToInt32(dr["qid"]);

                                    t.Questions.Add(new ModelIntake.Question
                                    {
                                        QID = m_lQID,
                                        QuestionTitle = dr["question"].ToString(),
                                        Responses = new List<ModelIntake.Response>()
                                    });
                                }
                            }
                        }
                    }
                }

                //add responses to the questions
                foreach (ModelIntake.Topic t in objModule.Topics)
                {
                    foreach (ModelIntake.Question q in t.Questions)
                    {

                        //loop the dataset
                        foreach (DataTable dt in ds.Tables)
                        {
                            foreach (DataRow dr in dt.Rows)
                            {
                                if (Convert.ToInt32(dr["tid"]) == t.TID && Convert.ToInt32(dr["qid"]) == q.QID)
                                {

                                    q.Responses.Add(new ModelIntake.Response
                                    {
                                        RID = Convert.ToInt32(dr["rid"]),
                                        ResponseValue = dr["response_value"].ToString(),
                                        StaticResponseText = dr["static_response"].ToString(),
                                        DisplayType = Convert.ToInt32(dr["display_type"]),
                                        ResponseType = Convert.ToInt32(dr["response_type"]),
                                        ResponseUnit = dr["unit"].ToString()
                                    });
                                }
                            }
                        }
                    }
                }

                //get instrument's score
                var s = new List<ModelIntake.Score>();
                DataSet dsScore = this.GetInstrumentScoresDS(strPatientID, lMID, lIntakeID);
                if(dsScore != null){
                    if (dsScore.Tables.Count > 0) {
                        
                        foreach (DataRow dr in dsScore.Tables[0].Rows) {
                            
                            string strSeries = String.Empty;
                            if(!dr.IsNull("series")){
                                strSeries = dr["series"].ToString();
                            }

                            string strScoreValue = String.Empty;
                            if (!dr.IsNull("score"))
                            {
                                strScoreValue = dr["score"].ToString();
                            }

                            string strInterpretation = String.Empty;
                            if (!dr.IsNull("interpretation"))
                            {
                                strInterpretation = dr["interpretation"].ToString();
                            }

                            string strDescription = String.Empty;
                            if (!dr.IsNull("description"))
                            {
                                strDescription = dr["description"].ToString();
                            }

                            s.Add(new ModelIntake.Score { 
                                Series = strSeries,
                                Description = strDescription,
                                ScoreValue = strScoreValue,
                                Interpretation = strInterpretation
                            });
                        }

                    }
                }

                objModule.Scores = s;
                obj = objModule;

            }
        }

        return obj;
    }

    /// <summary>
    /// Remove responses from previous time submitted
    /// not present in a new submittal of the instrument
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="strResponses"></param>
    /// <returns></returns>
    public bool RemovePrevResponses(string strPatientID,
                                        long lMID,
                                        long lIntakeID,
                                        string strResponses)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_vResponses", strResponses);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.RemovePrevResponses",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// Removes a specific RID from the intrument intake
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lRID"></param>
    /// <returns></returns>
    public bool RemoveResponse(string strPatientID,
                                    long lMID,
                                    long lIntakeID,
                                    long lRID)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nRID", lRID);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.RemoveResponse",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    //---------- SCORES

    /// <summary>
    /// Inserts Instrument score series
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lSeriesID"></param>
    /// <param name="lScore"></param>
    /// <param name="lInterpretID"></param>
    /// <returns></returns>
    public bool InsertInstrumentScore(string strPatientID,
                                    long lMID,
                                    long lIntakeID,
                                    long lSeriesID,
                                    long lScore,
                                    long lInterpretID)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        /*
            pi_vPatientID       in varchar2,
            pi_nIntakeID        in number,
            pi_nMID             in number,
            pi_nSeriesID        in number,
            pi_nScore           in number,
            pi_nInterpretID     in number,         
         */

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nSeriesID", lSeriesID);
        plist.AddInputParameter("pi_nScore", lScore);
        plist.AddInputParameter("pi_nInterpretID", lInterpretID);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.InsertInstrumentScore",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /////////////////////////////////////////////////////////////////////////////////////////
    /// <summary>
    /// Inserts Instrument score series
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lSeriesID"></param>
    /// <param name="lScore"></param>
    /// <param name="lInterpretID"></param>
    /// <param name="strDescription"></param>
    /// <returns></returns>
    public bool InsertInstrumentScore(string strPatientID,
                                long lMID,
                                long lIntakeID,
                                long lSeriesID,
                                double dScore,
                                string strInterpret,
                                string strDescription)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        /*
            pi_vPatientID       in varchar2,
            pi_nIntakeID        in number,
            pi_nMID             in number,
            pi_nSeriesID        in number,
            pi_nScore           in number,
            pi_vInterpret       in varchar2, 
            pi_vDescription     in varchar2, 
         */

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nSeriesID", lSeriesID);
        plist.AddInputParameter("pi_nScore", dScore);
        plist.AddInputParameter("pi_vInterpret", strInterpret);
        plist.AddInputParameter("pi_vDescription", strDescription);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.InsertInstrumentScore",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /////////////////////////////////////////////////////////////////////////////////////////
    /// <summary>
    /// Inserts Instrument score series
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lSeriesID"></param>
    /// <param name="lScore"></param>
    /// <param name="lInterpretID"></param>
    /// <param name="strDescription"></param>
    /// <returns></returns>
    public bool InsertInstrumentScore(string strPatientID,
                                long lMID,
                                long lIntakeID,
                                long lSeriesID,
                                long lScore,
                                string strInterpret,
                                string strDescription)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        /*
            pi_vPatientID       in varchar2,
            pi_nIntakeID        in number,
            pi_nMID             in number,
            pi_nSeriesID        in number,
            pi_nScore           in number,
            pi_vInterpret       in varchar2, 
            pi_vDescription     in varchar2, 
         */

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddInputParameter("pi_nSeriesID", lSeriesID);
        plist.AddInputParameter("pi_nScore", lScore);
        plist.AddInputParameter("pi_vInterpret", strInterpret);
        plist.AddInputParameter("pi_vDescription", strDescription);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.InsertInstrumentScore",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }


    /// <summary>
    /// Deletes Instrument Scores
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public bool DeleteInstrumentScore(string strPatientID,
                                      long lMID,
                                      long lIntakeID)
    {

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);

        /*
            pi_vPatientID       in varchar2,
            pi_nIntakeID        in number,
            pi_nMID             in number,
         */

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);

        m_BaseMstr.DBConn.ExecuteOracleSP("PCK_INSTRUMENT_RESPONSES.DeleteInstrumentScore",
                                          plist,
                                          out lStatusCode,
                                          out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public DataSet GetInstrumentScoresDS(string strPatientID, long lMID, long lIntakeID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);
        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);

        //
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetInstrumentScoresRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;
        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public DataSet GetInstrumentScoresDS(string strPatientID, long lMID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(m_BaseMstr.ASPSessionID,
                                                          m_BaseMstr.ClientIP,
                                                          m_BaseMstr.FXUserID);
        //add params for the DB stored procedure call
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nMID", lMID);

        //
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(m_BaseMstr.DBConn,
                                           "PCK_INSTRUMENT_RESPONSES.GetInstrumentScoresRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);

        //set the base master status code and status for display
        m_BaseMstr.StatusCode = lStatusCode;
        m_BaseMstr.StatusComment = strStatusComment;
        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public object GetInstrumentScoreObj(string strPatientID, long lMID) {
        
        //string strJSON = "{}";
        object oScore = new object { };

        DataSet ds = this.GetInstrumentScoresDS(strPatientID, lMID);
        if(ds != null){
            if (ds.Tables[0].Rows.Count > 0)
            {
                List<object> lstObj = new List<object>();

                //get list of series
                DataView vw = new DataView(ds.Tables[0]);
                DataTable distinctValues = vw.ToTable(true, "SERIES");

                string strMID = String.Empty;
                string strModule = String.Empty;


                foreach (DataRow drD in distinctValues.Rows) {

                    DataView vwSeries = new DataView(ds.Tables[0]);
                    vwSeries.RowFilter = "SERIES = " + drD["SERIES"].ToString();
                    DataTable dtSeries = vwSeries.ToTable();
                    List<object> lstObjData = new List<object>();

                    string strDescription = String.Empty;

                    foreach (DataRow dr in dtSeries.Rows) {
                        string strInterpretation = "N/A";

                        if (!dr.IsNull("mid")) {
                            strMID = dr["mid"].ToString();
                        }

                        if (!dr.IsNull("module"))
                        {
                            strModule = dr["module"].ToString();
                        }

                        if (!dr.IsNull("description"))
                        {
                            strDescription = dr["description"].ToString();
                        }
                        
                        if (!dr.IsNull("interpretation")) {
                            strInterpretation = dr["interpretation"].ToString();
                        }

                        decimal dScore = -1;
                        if(!dr.IsNull("score")){
                            dScore = Convert.ToDecimal(dr["score"]);
                        }

                        string strDateString = String.Empty;
                        double dDateMilliseconds = -1;
                        if (!dr.IsNull("date_completed")) {
                            strDateString = Convert.ToDateTime(dr["date_completed"]).ToShortDateString();
                            dDateMilliseconds = (Convert.ToDateTime(dr["date_completed"].ToString()) - new DateTime(1970, 1, 1)).TotalMilliseconds;
                        }

                        //only add a new score data object if the date is not null
                        if (true)
                        {
                            var d = new
                            {
                                date_string = strDateString,
                                x = dDateMilliseconds,
                                y = dScore,
                                interpretation = strInterpretation
                            };

                            lstObjData.Add(d); 
                        }
                    }

                    if (String.IsNullOrEmpty(strDescription)) { 
                        strDescription = "Series_" + drD["SERIES"].ToString();
                    }

                    var obj = new {
                        name = strDescription,
                        data = lstObjData
                    };
                    
                    lstObj.Add(obj);
                }

                var modScore = new { 
                    mid = strMID,
                    module = strModule,
                    scores = lstObj
                };

                //strJSON = JsonConvert.SerializeObject(modScore);
                oScore = modScore;
            }
        }
        return oScore;
    }

    /*
     * +--------------------------------------+
     * |     DATA MODELING CLASSES            |
     * +--------------------------------------+
     */

    /// <summary>
    /// Helper class for modeling the responses collection
    /// </summary>
    public class IntakeResponse {
        public IntakeResponse() { }
        public string PatientID { set; get; }
        public long MID { set; get; }
        public long TID { set; get; }
        public long QID { set; get; }
        public long RID { set; get; }
        public string Identifier { set; get; }
        public long ScoreValue { set; get; }
        public string ResponseValue { set; get; }
        public long ResponseType { set; get; }
    }

    /// <summary>
    /// 
    /// </summary>
    public class ModelIntake
    {

        //MODULE
        public class Module
        {
            public long MID { set; get; }
            public string ModuleTitle { set; get; }
            public List<Topic> Topics { set; get; }
            public List<Score> Scores { set; get; }
        }

        //TOPIC
        public class Topic
        {
            public long TID { set; get; }
            public string TopicTitle { set; get; }
            public List<Question> Questions { set; get; }
        }

        //QUESTION
        public class Question
        {
            public long QID { set; get; }
            public string QuestionTitle { set; get; }
            public List<Response> Responses { set; get; }
        }

        //RESPONSE
        public class Response
        {
            public long RID { set; get; }
            public string ResponseValue { set; get; }
            public long ScoreValue { set; get; }
            public string StaticResponseText { set; get; }
            public string ResponseUnit { set; get; }
            public long DisplayType { set; get; }
            public long ResponseType { set; get; }
        }

        //SCORE
        public class Score
        {
            public string Series { get; set; }
            public string Description { get; set; }
            public string ScoreValue { get; set; }
            public string Interpretation { get; set; }
        }
    }

}